<#========================================================================================#
#
# Script: To install 32bit/64bit DataLoader Single Tenant Services.
# Date: 19-Jul-24
#
# Author: JR
#
#========================================================================================#>

#========================================================================================#
# Change Log
#========================================================================================#

# 19-Jul-24 - JR - Script Created.

#========================================================================================#
# Begin Script
#========================================================================================#


#=========# Script predefining certain params. #==========#

## Sorting out running directory, this defines the directory where 0_SynergyInstall.ps1 regardless of if the script is being run in ISE, VS Code or just as a file.
if($MyInvocation.MyCommand.CommandType -eq 'ExternalScript'){
    $SynergyInstallScriptPath = (Split-Path -Path $MyInvocation.MyCommand.Source -Parent)
} # Running from File
if(Test-Path variable:psISE){
    $SynergyInstallScriptPath = (Split-Path -Path $psISE.CurrentFile.FullPath -Parent)
} # Running from ISE
if(Test-Path variable:psEditor){
    $SynergyInstallScriptPath = (Split-Path -Path $psEditor.GetEditorContext().CurrentFile.Path -Parent)
} # Running from Visual Studio Code

#Setting the location of the script to allow the . sourcing & running other files to work. 
Set-Location $SynergyInstallScriptPath

#Used to identify if 
$ComputerArchitecture = Get-WmiObject win32_operatingsystem | Select-Object -ExpandProperty osarchitecture

$32bitinstaller = (test-path ".\DataLoaderSetup.exe")
$64bitinstaller = (test-path ".\DataLoaderSetup64.exe")
$SecurityKey = (test-path ".\SecurityKey")


#=========# Script pre-checks #==========#

#Output location for logs and password files and the like.
$logOutput = "c:\ScriptLogs\DataEngine\DataLoaderServiceInstall\$(get-date -Format "MMddyyyy_HHmm")\"

#Starting Transcript Log
start-transcript -OutputDirectory $logOutput

Write-Host "Beginning Prechecks." -ForegroundColor Yellow

#Checking whether this is being run with elevated permissions.
if (!((New-Object Security.Principal.WindowsPrincipal([Security.Principal.WindowsIdentity]::GetCurrent())).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator))){
    throw "Script must be run as Admin."
}

#Confirming that we're running 64bit or 32bit windows.
if(($ComputerArchitecture -ne "64-bit") -and ($ComputerArchitecture -ne "32-bit")){
    throw "[Error] Unknown Architecture."
    Pause
}

#confirming that the installers exist.
if(($32bitinstaller -ne $true) -and ($64bitinstaller -ne $true)){
    throw "[Error] Cannot find installers."
    Pause

}

Write-Host "Prechecks Complete." -ForegroundColor Green


#=========# Locate Installers & Start install. #==========#

Write-Host "Beginning Installation" -ForegroundColor Yellow

Function Install-DataLoaderService {
    Param(
        [parameter(Mandatory = $True, Position = 0)]
        [String]$Installer,
        [parameter(Mandatory = $True, Position = 1)]
        [String]$ServiceName
    )

    Try{
        Write-Host "Starting Installer." -ForegroundColor Yellow
        
        Start-Process $Installer /silent # the /silent switch still brings the UI up but makes it so that it doesn't prompt for any config. It will install the app to the default location.

        Write-Host "Checking Service is installed." -ForegroundColor Yellow
        
        # This loop is just checking that the service is running as presumably the last thing the installer does is start the service.
        # This does mean that if you're using the executable to reinstall the service the service has to be stopped or uninstalled before you can run this script.
        [bool]$loop = $false
        [int]$timeout = 0
        while(!($loop)){
            $ServiceState = Get-Service | Where-Object{$_.name -eq $ServiceName}

            If($ServiceState.Status -eq "Running"){
                Write-Host "Service Installed and running." -ForegroundColor Green
                $loop = $true # This closes the loop.
            }
            Else{
                Write-Host "Service $($ServiceName) not running, waiting 10 seconds." -ForegroundColor Yellow
                Start-Sleep -Seconds 10

                # Adds 1 to the timeout int.
                $timeout ++
                # I've popped a timeout in so that this doesn't run indefinitely if something goes wrong with the install. it defaults to 3 minutes (18 iterations of 10 second waits.)
                if($timeout -gt 18){
                    Throw "Service $($ServiceName) not started. Loop timed out."
                }
            }
        }

    #Restarting service at the end to prompt the registry values.
    Restart-Service -Name "DataLoaderService64"
    }
    Catch{
        Throw "[Error] There was an issue installing the Data Loader Service. The returned error was '$($_.Exception.Message)'."
        Pause
    }
}

#Calling the above function for the appropriate architecture.
If ($ComputerArchitecture -eq "64-bit"){
    Install-DataLoaderService -Installer ".\DataLoaderSetup64.exe" -ServiceName "DataLoaderService64"
}
If ($ComputerArchitecture -eq "32-bit"){
    Install-DataLoaderService -Installer ".\DataLoaderSetup.exe" -ServiceName "DataLoaderService"
}

Write-Host "Installation Complete" -ForegroundColor Green


#=========# Getting security token. #==========#
$Token = Get-Content ".\SecurityKey"

Write-Host "Security Key is '$($Token)'." -ForegroundColor Cyan


#=========# Final Steps & Cleanup. #==========#
Write-Host "Script Complete" -ForegroundColor Green

Stop-Transcript